<?php

namespace App\Repositories\Website;

use App\Models\Client;
use App\Models\Movie;
use App\Models\RentalHistory;
use App\Models\Subscription;
use App\Models\SubscriptionTransactionLog;
use App\Models\TvShow;
use App\Models\User;
use App\Repositories\Interfaces\Website\SubscriptionRepositoryInterface;
use Illuminate\Support\Facades\Auth;
use App\Traits\ImageTrait;
use Carbon\Carbon;
use Exception;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Log;

class SubscriptionRepository implements SubscriptionRepositoryInterface
{
    use ImageTrait;
    // client
    public function create($plan, $trx_id, $payment_details, $billingInfo, $offline = false, $payment_method = 'stripe')
    {
        $status       = 1;
        if ($offline) {
            $payment_method  = 'offline';
            $payment_details = json_encode(['payment_type' => 'offline']);
            $status = 0;

            if (isset($billingInfo['transaction_file'])) {
                if (isset($billingInfo['transaction_file']) && $billingInfo['transaction_file'] instanceof UploadedFile) {
                    $requestImage = $billingInfo['transaction_file'];
                    $response = $this->saveImage($requestImage, '_offline_payment_');
                    $file = $response['images'];
                }
            }
        }

        $client       = isset($plan['client_id']) ? Client::where('id', $plan['client_id'])->first() : Auth::user()->client;
        $is_recurring = 0;
        $expire_date  = now();

        if ($plan->type == 'monthly') {
            $expire_date  = now()->addDay($plan->day);
            $is_recurring = 1;
        }
        if ($plan->type == 'annually') {
            $expire_date  = now()->addDay($plan->day);
            $is_recurring = 1;
        }

        $subscription = Subscription::where('client_id', $client->id)->where('status', 1)->first();
        if ($subscription) {
            $subscription->status = 3;
            $subscription->save();
        }

        User::where('client_id', $client->id)->update(['subscription_id' => $plan->id]);

        $data         = [
            'client_id'         => $client->id,
            'plan_id'           => $plan->id ?? $plan['plan_id'],
            'is_recurring'      => $is_recurring,
            'status'            => $status,
            'purchase_date'     => now(),
            'expire_date'       => $expire_date,
            'price'             => $plan->price,
            'package_type'      => $plan->type,
            'profile_limit'     => $plan->screen,
            'post_limit'        => $plan->post_limit,
            'post_remaining'    => $plan->post_limit,
            'profile_remaining' => $plan->profile_limit,
            'team_limit'        => $plan->team_limit,
            'trx_id'            => $trx_id,
            'payment_method'    => $payment_method,
            'payment_details'   => isset($file) ? json_encode($file) : $payment_details,
            'client'            => Client::find($client->id),
            'subject'           => __('package_subscription_confirmation'),
        ];

        session()->forget('billing_info');

        $log          = SubscriptionTransactionLog::create([
            'description' => __('you purchased ') . $plan->name . __('package_successfully'),
            'client_id'   => $client->id,
        ]);

        return Subscription::create($data);
    }

    public function cancelSubscription($id)
    {
        $subscription              = Subscription::find($id);

        $payment_method            = $subscription->payment_method;

        if ($payment_method == 'stripe') {
            $this->cancelStripe($subscription);
        } elseif ($payment_method == 'paddle') {
            $this->cancelPaddle($subscription);
        } elseif ($payment_method == 'paypal') {
            $this->cancelPaypal($subscription);
        }

        $subscription->canceled_at = now();
        $subscription->status      = 2;
        $log                       = SubscriptionTransactionLog::create([
            'description' => 'You cancel Your Subscription',
            'client_id'   => auth()->user()->client_id,
        ]);
        $subscription->save();

        return $subscription;
    }

    public function cancelStripe($subscription)
    {
        $stripe_subscript_id = getArrayValue('subscription', $subscription->payment_details);
        $response            = [];
        if ($stripe_subscript_id) {
            $headers  = [
                'Authorization' => 'Basic ' . base64_encode(setting('stripe_secret') . ':'),
                'Content-Type'  => 'application/x-www-form-urlencoded',
            ];

            $data     = [
                'invoice_now' => 'false',
            ];
            $response = httpRequest('https://api.stripe.com/v1/subscriptions/' . $stripe_subscript_id, $data, $headers, true, 'DELETE');
        }

        return $response;
    }

    public function cancelPaddle($subscription)
    {
        $transaction_id  = $subscription->payment_details['transaction_id'];

        $headers         = [
            'Authorization' => 'Bearer ' . setting('paddle_api_key'),
        ];
        if (setting('is_paddle_sandbox_mode_activated')) {
            $base_url = 'https://sandbox-api.paddle.com/';
        } else {
            $base_url = 'https://api.paddle.com/';
        }
        $data            = [
            'effective_from' => 'next_billing_period',
        ];
        $response        = httpRequest($base_url . "transactions/$transaction_id", $data, $headers, false, 'GET');
        $subscription_id = $response['data']['subscription_id'];

        return httpRequest($base_url . "subscriptions/$subscription_id/cancel", $data, $headers);
    }

    public function cancelPaypal($subscription)
    {
        if (setting('is_paypal_sandbox_mode_activated')) {
            $base_url = 'https://api-m.sandbox.paypal.com';
        } else {
            $base_url = 'https://api-m.paypal.com';
        }
        $paypal_subscription_id = $subscription->payment_details['id'];
        $headers                = [
            'Content-Type'  => 'application/json',
            'Authorization' => $this->paypalTokenGenerator($base_url),
        ];

        $data                   = [
            'reason' => 'stopped by admin',
        ];

        return httpRequest($base_url . '/v1/billing/subscriptions/' . $paypal_subscription_id . '/cancel', $data, $headers);
    }

    public function paypalTokenGenerator($base_url): string
    {
        // generate access token
        $headers  = [
            'Content-Type'  => 'application/x-www-form-urlencoded',
            'Authorization' => 'Basic ' . base64_encode(setting('paypal_client_id') . ':' . setting('paypal_client_secret')),
        ];
        $data     = [
            'grant_type' => 'client_credentials',
        ];
        $response = httpRequest($base_url . '/v1/oauth2/token', $data, $headers, true);

        return $response['token_type'] . ' ' . $response['access_token'];
    }

    public function subscribeListStatus($request, $id)
    {
        $subscribe         = Subscription::findOrfail($id);
        $subscribe->status = $request['status'];
        if ($request['status'] == 2) {
            $payment_method         = $subscribe->payment_method;
            if ($payment_method == 'stripe') {
                $this->cancelStripe($subscribe);
            } elseif ($payment_method == 'paddle') {
                $this->cancelPaddle($subscribe);
            } elseif ($payment_method == 'paypal') {
                $this->cancelPaypal($subscribe);
            }
            $subscribe->canceled_at = now();
        }
        $status            = __('pending');
        if ($request['status'] == 1) {
            $status = __('active');
        } elseif ($request['status'] == 2) {
            $status = __('cancelled');
        } elseif ($request['status'] == 3) {
            $status = __('rejected');
        }
        $msg               = __('subscription_status_updated', ['status' => $status]);

        $log               = SubscriptionTransactionLog::create([
            'description' => 'Admin ' . $status . ' your plan',
            'client_id'   => $subscribe->client_id,
        ]);

        return $subscribe->save();
    }

    public function stopRecurring($id)
    {
        $subscription               = Subscription::find($id);

        $payment_method             = $subscription->payment_method;

        if ($payment_method == 'stripe') {
            $this->cancelStripe($subscription);
        } elseif ($payment_method == 'paddle') {
            $this->cancelPaddle($subscription);
        } elseif ($payment_method == 'paypal') {
            $this->cancelPaypal($subscription);
        }
        $cancel_date                = Carbon::parse($subscription->purchase_date);

        if ($subscription->package_type == 'monthly') {
            $cancel_date  = $cancel_date->addDay();
        }
        if ($subscription->package_type == 'annually') {
            $cancel_date  = $cancel_date->addYear();
        }

        $subscription->canceled_at  = $cancel_date;
        $subscription->is_recurring = 0;

        if (auth()->user()->user_type == 'admin') {
            $log = SubscriptionTransactionLog::create([
                'description' => 'admin stop your recurring',
                'client_id'   => $subscription->client_id,
            ]);
        } else {
            $log = SubscriptionTransactionLog::create([
                'description' => 'you stop your recurring',
                'client_id'   => $subscription->client_id,
            ]);
        }

        $subscription->save();

        return $subscription;
    }

    public function rentVideo($request, $type)
    {
        try {

            if ($request['video_type'] == 'movie') {
                $video = Movie::where('id', $request['plan_id'])->first();
            } elseif ($request['video_type'] == 'series') {
                $video = TvShow::where('id', $request['plan_id'])->first();
            }

            if (isset($request['transaction_file'])) {
                if (isset($request['transaction_file']) && $request['transaction_file'] instanceof UploadedFile) {
                    $requestImage = $request['transaction_file'];
                    $response = $this->saveImage($requestImage, '_offline_payment_');
                    $file = $response['images'];
                }
            }

            $data = [
                'user_id'        => auth()->id() ?? $request['user_id'],
                'video_id'       => $video->id,
                'video_type'     => $request['video_type'],
                'transaction_id' => $request['transaction_key'] ?? $request['trx_id'],
                'price'          => $video->rental_price,
                'duration'       => $video->rental_duration,
                'type'           => $type,
                'file'           => isset($file) ? json_encode($file) : null,
                'status'         => ($type == 'offline') ? 0 : 1,
                'created_at'     => Carbon::now(),
                'updated_at'     => Carbon::now(),
            ];

            // Only insert if user+video doesn’t already exist
            return RentalHistory::firstOrCreate(
                [
                    'user_id' => $data['user_id'],
                    'video_id' => $data['video_id'],
                    'video_type' => $data['video_type'],
                ],
                $data
            );
        } catch (Exception) {
            return false;
        }
    }
}
