<?php

namespace App\Repositories\Admin;

use App\Models\MovieFile;
use App\Models\Package;
use App\Models\PlanCredential;
use App\Models\Star;
use App\Repositories\Interfaces\Admin\PackageInterface;
use Carbon\Carbon;
use App\Traits\ImageTrait;
use Illuminate\Support\Str;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Log;

class PackageRepository implements PackageInterface
{
    use ImageTrait;

    public function all()
    {
        return Package::get();
    }

    public function find($id) {
        return Package::where('id', $id)->first();
    }

    public function store($request)
{
    try {
        $data = [
            'name'      => $request['name'],
            'day'       => $request['day'],
            'price'     => $request['price'],
            'discount'  => $request['discount'],
            'screens'   => $request['screens'],
            'type'      => $request['type'],
            'revenue_cat_id' => $request['revenue_cat_id'],
            'status'    => $request['status'],
            'created_at'=> Carbon::now(),
            'updated_at'=> Carbon::now(),
        ];

        $package = Package::create($data);

        $paymentData = $request;
        $paymentData['id'] = $package->id;

        $this->createStripePlan($paymentData);
        $this->createPaypalPlan($paymentData);
        $this->createPaddlePlan($paymentData);
        $this->createRazorPayPlan($paymentData);

        return $package;
    } catch (\Exception $e) {
        // Log error for debugging
        Log::error('Package creation failed', ['error' => $e->getMessage()]);
        return false;
    }
}
function arrayCheck($key, $array)
{
    return isset($array[$key]) && !empty($array[$key]);
}

   public function update($request, $id)
{
     try {
        // Step 1: Update package info
        $data = [
            'name'      => $request['name'],
            'day'       => $request['day'],
            'price'     => $request['price'],
            'discount'  => $request['discount'] ?? 0,
            'screens'   => $request['screens'],
            'type'      => $request['type'],
            'revenue_cat_id' => $request['revenue_cat_id'],
            'status'    => $request['status'],
            'updated_at'=> Carbon::now(),
        ];

        $updated = Package::where('id', $id)->update($data);

        // Step 2: Add plan ID to request-like array for credential methods
        $request['id'] = $id;

        // Step 3: Handle plan credentials
        $this->createStripePlan($request);
        $this->createPaypalPlan($request);
        $this->createPaddlePlan($request);
        $this->createRazorPayPlan($request);

        return $updated;
    } catch (\Exception $e) {
        Log::error('Package update failed', ['error' => $e->getMessage()]);
        return false;
    }
}


    public function createStripePlan($data)
    {
        if (arrayCheck('stripe', $data)) {
            $package = PlanCredential::where('plan_id', $data['id'])->where('title', 'stripe')->first();

            if ($package) {
                $package->value = $data['stripe'];
                $package->save();
            } else {
                PlanCredential::create([
                    'plan_id' => $data['id'],
                    'title'   => 'stripe',
                    'value'   => $data['stripe'],
                ]);
            }
        }

        return null;
    }

    public function createPaypalPlan($data)
    {
        if (arrayCheck('paypal', $data)) {
            $package = PlanCredential::where('plan_id', $data['id'])->where('title', 'paypal')->first();

            if ($package) {
                $package->value = $data['paypal'];
                $package->save();
            } else {
                PlanCredential::create([
                    'plan_id' => $data['id'],
                    'title'   => 'paypal',
                    'value'   => $data['paypal'],
                ]);
            }
        }

        return null;
    }

    public function createPaddlePlan($data)
    {
        if (arrayCheck('paddle', $data)) {
            $package = PlanCredential::where('plan_id', $data['id'])->where('title', 'paddle')->first();

            if ($package) {
                $package->value = $data['paddle'];
                $package->save();
            } else {
                PlanCredential::create([
                    'plan_id' => $data['id'],
                    'title'   => 'paddle',
                    'value'   => $data['paddle'],
                ]);
            }
        }

        return null;
    }

    public function createRazorPayPlan($data)
    {
        if (arrayCheck('razor_pay', $data)) {
            $package = PlanCredential::where('plan_id', $data['id'])->where('title', 'razor_pay')->first();
            if ($package) {
                $package->value = $data['razor_pay'];
                $package->save();
            } else {
                PlanCredential::create([
                    'plan_id' => $data['id'],
                    'title'   => 'razor_pay',
                    'value'   => $data['razor_pay'],
                ]);
            }
        }

        return null;
    }

    public function getPGCredential($plan_id, $title)
    {
        try {
            $package = PlanCredential::where('plan_id', $plan_id)->where('title', $title)->value('value');
            return $package;
        } catch (\Exception $e) {
            return false;
        }
    }
}