<?php

namespace App\Http\Controllers\Api\Mobile;

use App\Http\Controllers\Controller;
use App\Models\Country;
use App\Models\Movie;
use App\Models\Star;
use App\Traits\Mobile\ApiResponseTrait;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class StarController extends Controller
{
    use ApiResponseTrait;

    public function artists(Request $request)
    {
        $artistFelids = function ($item) {
            $item->star_image = apiImage(optional(json_decode($item->star_image))->original_image, 'stars');
            return $item;
        };

        $artists = Star::query()
            ->where('status', 1)
            ->when($request->name, function ($query, $name) {
                $query->where('star_name', 'like', '%' . $name . '%');
            })
            ->when($request->country, function ($query, $country) {
                $query->where('country_id', $country);
            })
         
            ->paginate(20)
            ->through($artistFelids);
        $data = [
            'artists' => $artists,
            'filter' => [
                'countries' => Country::where('status', 1)->get(),
                'movie_industries' => collect([
                    (object) ['id' => 1, 'name' => 'Hollywood', 'status' => 1],
                    (object) ['id' => 2, 'name' => 'Bollywood', 'status' => 1],
                    (object) ['id' => 3, 'name' => 'Tollywood', 'status' => 1],
                ])
            ]
        ];
        return $this->responseWithSuccess($data, __('artists_list_retrieved_successfully'));
    }

    public function artist_details(Request $request)
    {

        $artistFelids = function ($item) {
            $item->star_image = optional(json_decode($item->star_image))->original_image;
            return $item;
        };

        $movieFelids = function ($item) {
            $item->thumbnail = optional(json_decode($item->thumbnail))->original_image;
            $item->poster = optional(json_decode($item->poster))->original_image;
            return $item;
        };

        $artist = Star::where('id', $request->id)->first();
        if (!$artist) {
            return $this->responseWithError(__('artist_not_found'), 400);
        }
        $movies = Movie::whereJsonContains('stars', (string) $artist->id)->get()->transform($movieFelids);
        $data = [
            'details' => $artistFelids($artist),
            'movies' => $movies
        ];
        return $this->responseWithSuccess($data, __('artist_details_retrieved_successfully'));
    }

    public function artist_filter(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string'
        ]);
        if ($validator->fails()) {
            return $this->responseWithValidation($validator->errors(), 422);
        }
        $artists = Star::where('star_name', 'LIKE', '%' . $request->name . '%')->get();

        if ($artists->isEmpty()) {
            return $this->responseWithError(__('no_artists_found_matching_name'), 404);
        }

        return $this->responseWithSuccess($artists, __('artists_filtered_successfully'));
    }
}
