<?php
namespace App\Http\Controllers\Admin;
use App\Http\Controllers\Controller;
use App\Repositories\Admin\PageRepository;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Log;

class PageController extends Controller
{
    protected $page;

    public function __construct(PageRepository $page)
    {
        $this->page = $page;
    }

    public function index()
    {
        try {
            $data = [
                'pages' => $this->page->all([
                    'paginate' => 10,
                ]),
            ];

            return view('admin.website.page.index', $data);

        } catch (\Exception $e) {
            Log::error('Error fetching pages: ' . $e->getMessage());
            session()->flash('toastr_danger', $e->getMessage());
            return back();
        }
    }

    public function create()
    {
        try {
            return view('admin.website.page.create');
        } catch (\Exception $e) {
            session()->flash('toastr_danger', $e->getMessage());

            return back();
        }
    }

    public function store(Request $request): \Illuminate\Http\JsonResponse
    {
        if (isDemoMode()) {
            $data = [
                'status'    => 'error',
                'message'   => 'This action is disabled in demo mode.',
                'title'     => __('error'),
                'is_reload' => false,
            ];
            return response()->json($data);
        }
        DB::beginTransaction();
        try {
            $this->page->store($request->all());
            session()->flash('toastr_success', __('create_successful'));
            DB::commit();

            return response()->json([
                'success' => __('create_successful'),
                'route' => route('pages.index'),
            ]);

        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json(['error' => $e->getMessage()]);
        }
    }

    public function edit($id, Request $request)
    {
        
        try {
            $lang = $request->lang ?? app()->getLocale();
            $data = [
                'languages' => activeLanguages(),
                'lang' => $lang,
                'page' => $this->page->get($id),
                'page_language' => $this->page->getByLang($id, $lang),
            ];

            return view('admin.website.page.edit', $data);
        } catch (\Exception $e) {
            session()->flash('toastr_danger', $e->getMessage());

            return back();
        }
    }

    public function update(Request $request, $id): \Illuminate\Http\JsonResponse
    {
        if (isDemoMode()) {
            $data = [
                'status'    => 'error',
                'message'   => 'This action is disabled in demo mode.',
                'title'     => __('error'),
                'is_reload' => false,
            ];
            return response()->json($data);
        }
        DB::beginTransaction();
        try {
            $this->page->update($request->all(), $id);
            session()->flash('toastr_success', __('update_successful'));
            DB::commit();

            return response()->json([
                'success' => __('update_successful'),
                'route' => route('pages.index'),
            ]);
        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json(['error' => $e->getMessage()]);
        }
    }

    public function destroy($id)
    {
        if (isDemoMode()) {
            $data = [
                'status'    => 'error',
                'message'   => 'This action is disabled in demo mode.',
                'title'     => __('error'),
                'is_reload' => false,
            ];
            return response()->json($data);
        }
        try {
            $this->page->destroy($id);
            session()->flash('toastr_success', 'delete_successful');

            $data = [
                'status' => 'success',
                'message' => __('delete_successful'),
                'title' => __('success'),
            ];

            return response()->json($data);
        } catch (\Exception $e) {
            $data = [
                'status' => 'danger',
                'message' => $e->getMessage(),
                'title' => __('error'),
            ];

            return response()->json($data);
        }
    }

    public function statusChange(Request $request): \Illuminate\Http\JsonResponse
    {
        if (isDemoMode()) {
            $data = [
                'status'    => 'error',
                'message'   => 'This action is disabled in demo mode.',
                'title'     => __('error'),
                'is_reload' => false,
            ];
            return response()->json($data);
        }
        try {
            $this->page->status($request->all());
            $data = [
                'status' => 200,
                'message' => __('update_successful'),
                'title' => 'success',
            ];

            return response()->json($data);
        } catch (\Exception $e) {
            $data = [
                'status' => 400,
                'message' => $e->getMessage(),
                'title' => 'error',
            ];

            return response()->json($data);
        }
    }
}
