<?php

namespace App\DataTables\Admin;

use App\Models\SubscriptionTransactionLog;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Services\DataTable;
use Yajra\DataTables\EloquentDataTable;

class TransactionLogDataTable extends DataTable
{
    public function dataTable(QueryBuilder $query): EloquentDataTable
    {
        return (new EloquentDataTable($query))
            ->addIndexColumn()
            ->addColumn('description', function ($subscription) {
                return $subscription->description;
            })
            ->addColumn('transaction_time', function ($subscription) {
                return $subscription->created_at;
            })
            ->addColumn('client', function ($subscription) {
                return $subscription->client->first_name.' '.$subscription->client->last_name;
            })->setRowId('id');
    }

    public function query(): QueryBuilder
    {
        return SubscriptionTransactionLog::when($this->request->search['value'] ?? false, function ($query) {
            $search = $this->request->search['value'];
            $query->where('trx_id', 'like', "%$search%");
        })
            ->latest()
            ->newQuery();
    }

    public function html(): HtmlBuilder
    {
        return $this->builder()
            ->columns($this->getColumns())
            ->minifiedAjax()
            ->orderBy(1)
            ->selectStyleSingle()
            ->setTableAttribute('style', 'width:99.8%')
            ->footerCallback('function ( row, data, start, end, display ) {

                $(".dataTables_length select").addClass("form-select form-select-lg without_search mb-3");
                selectionFields();
            }')
            ->parameters([
                'dom'        => 'Blfrtip',
                'buttons'    => [
                    [],
                ],
                'lengthMenu' => [[10, 25, 50, 100, 250], [10, 25, 50, 100, 250]],
                'language'   => [
                    'searchPlaceholder' => __('search'),
                    'lengthMenu'        => '_MENU_ ' . __('transaction_per_page'),
                    'search'            => '',
                ],
            ]);
    }

    public function getColumns(): array
    {
        return [
            Column::computed('id')->data('DT_RowIndex')->title('#')->width(10),
            Column::computed('description')->title(__('description')),
            Column::computed('transaction_time')->title(__('transaction_time')),
            Column::computed('client')->title(__('client_name'))
                ->exportable(false)
                ->printable(false)
                ->searchable(false)->addClass('action-card')->width(10),
        ];
    }

    protected function filename(): string
    {
        return 'Currency_' . date('YmdHis');
    }

}
