<?php

namespace App\DataTables\Admin;

use App\Models\RentalHistory;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Services\DataTable;
use Yajra\DataTables\EloquentDataTable;

class RentalHistoryDataTable extends DataTable
{
    public function dataTable(QueryBuilder $query): EloquentDataTable
    {
        return (new EloquentDataTable($query))
            ->addIndexColumn()
            ->addColumn('user', function ($history) {
                return $history->user->first_name.' '.$history->user->last_name;
            })
            ->addColumn('movie_name', function ($history) {
                return $history->video_type == 'movie' ? $history->movie->title : $history->series->title;
            })
            ->addColumn('rent_time', function ($history) {
                return $history->created_at;
            })
            ->addColumn('transaction_key', function ($history) {
                return $history->transaction_id;
            })
            ->addColumn('price', function ($history) {
                return get_currency($history->price);
            })
            ->addColumn('type', function ($history) {
                return ucwords($history->type);
            })
            ->addColumn('file', function ($history) {
                return view('admin.subscription.column.rent_file', compact('history'));
            })
            ->addColumn('action', function ($history) {
                return view('admin.subscription.column.rent_action', compact('history'));
            })
            ->setRowId('id');
    }

    public function query(): QueryBuilder
    {
        return RentalHistory::when($this->request->search['value'] ?? false, function ($query) {
            $search = $this->request->search['value'];
            $query->where('transaction_id', 'like', "%$search%");
        })
            ->latest()
            ->newQuery();
    }

    public function html(): HtmlBuilder
    {
        return $this->builder()
            ->columns($this->getColumns())
            ->minifiedAjax()
            ->orderBy(1)
            ->selectStyleSingle()
            ->setTableAttribute('style', 'width:99.8%')
            ->footerCallback('function ( row, data, start, end, display ) {

                $(".dataTables_length select").addClass("form-select form-select-lg without_search mb-3");
                selectionFields();
            }')
            ->parameters([
                'dom'        => 'Blfrtip',
                'buttons'    => [
                    [],
                ],
                'lengthMenu' => [[10, 25, 50, 100, 250], [10, 25, 50, 100, 250]],
                'language'   => [
                    'searchPlaceholder' => __('search'),
                    'lengthMenu'        => '_MENU_ ' . __('rental_history_per_page'),
                    'search'            => '',
                ],
            ]);
    }

    public function getColumns(): array
    {
        return [
            Column::computed('id')->data('DT_RowIndex')->title('#')->width(10),
            Column::computed('user')->title(__('user')),
            Column::computed('movie_name')->title(__('video_name')),
            Column::computed('rent_time')->title(__('rent_time')),
            Column::computed('transaction_key')->title(__('transaction_key')),
            Column::computed('type')->title(__('type')),
            Column::computed('price')->title(__('price')),
            Column::computed('file')->title(__('file')),
            Column::computed('action')->title(__('action'))
                ->exportable(false)
                ->printable(false)
                ->searchable(false)->addClass('action-card')->width(10),
        ];
    }

    protected function filename(): string
    {
        return 'Currency_' . date('YmdHis');
    }

}
