<?php

namespace App\DataTables\Admin;

use App\Models\CustomerQuery;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Services\DataTable;
use Yajra\DataTables\EloquentDataTable;

class CustomerQueryDataTable extends DataTable
{
    public function dataTable(QueryBuilder $query): EloquentDataTable
    {
        return (new EloquentDataTable($query))
            ->addIndexColumn()
            ->addColumn('status', function ($customerQuery) {
                return view('admin.customer_queries.partials.status', compact('customerQuery'));
            })
            ->addColumn('option', function ($customerQuery) {
                return view('admin.customer_queries.partials.action', compact('customerQuery'));
            })
            ->setRowId('id');
    }

    public function query(): QueryBuilder
    {
        return CustomerQuery::when($this->request->search['value'] ?? false, function ($query) {
            $search = $this->request->search['value'];
            $query->where('name', 'like', "%$search%")
                ->orWhere('email', 'like', "%$search%")
                ->orWhere('subject', 'like', "%$search%")
                ->orWhere('message', 'like', "%$search%");
        })->latest()->newQuery();
    }

    public function html(): HtmlBuilder
    {
        return $this->builder()
            ->columns($this->getColumns())
            ->minifiedAjax()
            ->orderBy(1)
            ->selectStyleSingle()
            ->setTableAttribute('style', 'width:99.8%')
            ->footerCallback('function ( row, data, start, end, display ) {
                $(".dataTables_length select").addClass("form-select form-select-lg without_search mb-3");
                selectionFields();
            }')
            ->parameters([
                'dom' => 'Blfrtip',
                'buttons' => [],
                'lengthMenu' => [[10, 25, 50, 100, 250], [10, 25, 50, 100, 250]],
                'language' => [
                    'searchPlaceholder' => __('search'),
                    'lengthMenu' => '_MENU_ ' . __('customer_query_per_page'),
                    'search' => '',
                ],
            ]);
    }

    public function getColumns(): array
    {
        return [
            Column::computed('id')->data('DT_RowIndex')->title('#')->width(10),
            Column::make('name')->title(__('Name')),
            Column::make('email')->title(__('Email')),
            Column::make('subject')->title(__('Subject')),
            Column::computed('status')->title(__('Status')),
            Column::computed('option')->title(__('Action'))->exportable(false)->printable(false),
        ];
    }

    protected function filename(): string
    {
        return 'Customer_Queries_' . date('YmdHis');
    }
}
